<?php
/* --------------------------------------------------------------
   FetchAllParcelServicesAction.php 2020-04-09
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Api\ParcelService\Actions;

use Gambio\Admin\ParcelService\Interfaces\ParcelServiceService;
use Gambio\Api\Application\Responses\CreateApiMetaDataTrait;
use Gambio\Api\Application\Responses\ResponseDataTrimmerTrait;
use Gambio\Api\ParcelService\ParcelServiceApiRequestParser;
use Slim\Http\Response;
use Slim\Http\ServerRequest as Request;

/**
 * Class FetchAllParcelServicesAction
 *
 * @package Gambio\Api\ParcelService\Actions
 */
class FetchAllParcelServicesAction
{
    use CreateApiMetaDataTrait;
    use ResponseDataTrimmerTrait;
    
    /**
     * @var ParcelServiceApiRequestParser
     */
    private $requestParser;
    
    /**
     * @var ParcelServiceService
     */
    private $service;
    
    
    /**
     * FetchAllParcelServicesAction constructor.
     *
     * @param ParcelServiceApiRequestParser $requestParser
     * @param ParcelServiceService          $service
     */
    public function __construct(
        ParcelServiceApiRequestParser $requestParser,
        ParcelServiceService $service
    ) {
        $this->requestParser = $requestParser;
        $this->service       = $service;
    }
    
    
    /**
     * @param Request  $request
     * @param Response $response
     * @param array    $args
     *
     * @return Response
     */
    public function __invoke(Request $request, Response $response, array $args): Response
    {
        $criteria = $this->requestParser->getCriteriaFromRequest($request);
        $fields   = $this->requestParser->getFieldsFromRequest($request);
        
        $parcelServices = $this->service->getAllParcelServices($criteria,
                                                               $this->requestParser->getPaginationFromRequest($request));
        $totalItems     = $this->service->getTotalCountOfParcelServices($criteria);
        $metaData       = $this->createApiCollectionMetaData($this->requestParser->getPageFromRequest($request),
                                                             $this->requestParser->getPerPageFromRequest($request),
                                                             $totalItems,
                                                             $this->requestParser->getResourceUrlFromRequest($request),
                                                             $request->getQueryParams());
        
        $responseData = $parcelServices;
        if (count($fields) > 0) {
            $collectionData = json_decode(json_encode($parcelServices), true);
            $responseData   = $this->trimCollectionData($collectionData, $fields);
        }
        
        return $response->withJson([
                                       'data'  => $responseData,
                                       '_meta' => $metaData,
                                   ]);
    }
}